#include "sn_debug.h"
#include "sn_manager.h"
#include <stdio.h>
#include "sn_packet.h"

// This is the file into which to compile the static data members
SN_Manager* SN_Manager::self_ptr;
set<SN_Process*> SN_Manager::processes;

// ----------------------------------------------------------------------------

SN_Manager::SN_Manager(int num_processes)
{
	// Initialize statistics
	round = 0;
	num_messages_sent = 0;
	num_messages_received = 0;
	num_messages_dropped = 0;

	// Number of processes + 1 for the manager
	MS_Init(num_processes + 1);

	// Store a pointer to ourselves so that the static Start_Sim_Process member knows
	// the manager object
	self_ptr = this;

	// Register ourselves with the infrastructure.
	// Make the infrastructure call us, so we can then shuttle messages
	// to the processes
	DEBUG_2("Registered manager (Process ID %d)\n", MANAGER_PROCESS_ID);
	MS_Regnode(MANAGER_PROCESS_ID,SN_Manager::Start_Sim_Process, NULL);
}

// ----------------------------------------------------------------------------

SN_Manager::~SN_Manager()
{
}

// ----------------------------------------------------------------------------

void SN_Manager::Start_Sim_Process(int process_id_to_start,MS_Msgbox *m,void *data)
{
	DEBUG_2("Manager: Got start message from MS for index %d\n", process_id_to_start);

	if (process_id_to_start == 0)
	{
		DEBUG_2("Manager: Index 0 is the manager, so we wait until the processes have started...\n");
	}

	static int num_processes_started = 0;

	// See if we need to start a process
	set<SN_Process*>::iterator a_process;

	for(a_process = processes.begin(); a_process != processes.end(); a_process++)
	{
		if ((*a_process)->process_id == process_id_to_start)
		{
			num_processes_started++;

			DEBUG_2("Manager: Starting process %d (of %d)\n",num_processes_started, processes.size());

			(*a_process)->Run(m);
			return;
		}
	}

	// Busy wait until all the processes are started
	while (num_processes_started < processes.size()) { }

	DEBUG_2("Manager: All the processes have started, so the manager will start now.\n");

	DEBUG("Manager: Press enter to continue...\n");
	getchar();

	self_ptr->Run(m);
}

// ----------------------------------------------------------------------------

void SN_Manager::Start_Sim()
{
	set<SN_Process*>::iterator a_process;

	for(a_process = processes.begin(); a_process != processes.end(); a_process++)
		(*a_process)->Set_Start_State();

	MS_Start ();
}

// ----------------------------------------------------------------------------

void SN_Manager::Run(MS_Msgbox *in_m)
{
	char end_char;

	m = in_m;

	// The manager always ships messages...
	while(1)
	{
		round++;

		Start_Round();

		DEBUG("Manager: Press enter to continue...\n");
		getchar();

		Get_Messages();

		DEBUG("Manager: Press enter to continue...\n");
		getchar();

		Drop_Messages();

		Send_Messages();

		DEBUG("Manager: Press enter to continue...\n");
		getchar();

		End_Round();

		printf("Manager: Press enter to continue (Or enter '0' to end)...\n");
		end_char = getchar();

		if (end_char == '0')
			End_Simulation();
	}
}

// ----------------------------------------------------------------------------

void SN_Manager::Start_Round()
{
	DEBUG_2("Manager: Starting round %d...\n", round);

	set<SN_Process*>::iterator a_process;

	for(a_process = processes.begin(); a_process != processes.end(); a_process++)
	{
		SN_Packet start_msg;
		start_msg.type = SN_Packet::START_ROUND_MSG;
		start_msg.from = SN_Manager::MANAGER_PROCESS_ID;
		start_msg.to = (*a_process)->process_id;

		m->send((*a_process)->process_id, &start_msg, sizeof(SN_Packet));
	}
}

// ----------------------------------------------------------------------------

void SN_Manager::Get_Messages()
{
	DEBUG_2("Manager: Receiving messages for round %d...\n", round);

	int num_procs_done_sending = 0;

	while (num_procs_done_sending != processes.size())
	{
		SN_Packet incoming_packet;
		int unused_sender_process_id;

		m->recv_any(&incoming_packet, sizeof(SN_Packet), &unused_sender_process_id);

		if (incoming_packet.type == SN_Packet::DONE_SENDING_MSG)
		{
			num_procs_done_sending++;
			// Skip to next incoming message
			continue;
		}

		DEBUG_3("Manager: Received message from process %d to process %d\n",
			incoming_packet.from, incoming_packet.to);

		// If a message was already sent on this channel in this round...
		if (channels.find(make_pair(incoming_packet.from,incoming_packet.to)) !=
			channels.end())
		{
			DEBUG("Received two messages in a single round from process");
			DEBUG("%d to process %d\n",incoming_packet.from,incoming_packet.to);
			exit(1);
		}
		else
		{
			Link new_link = make_pair(incoming_packet.from,incoming_packet.to);
			channels.insert(make_pair(new_link,incoming_packet.message));
			num_messages_sent++;
		}
	}
}

// ----------------------------------------------------------------------------

void SN_Manager::Drop_Messages()
{
	// should be overridden in an inherited class
	// Example code
/*
	printf("Manager: Dropping messages for round %d...\n", round);

	map<Link,Message>::iterator a_channel;
	map<Link,Message>::iterator deleted_channel;

	for(a_channel = temp_channels->begin(); a_channel != temp_channels->end(); a_channel++)
	{
		Link from_to = a_channel->first;

		int from_process_id = from_to.first;
		int to_process_id = from_to.second;

		if ((from_process_id == 1) && (to_process_id == 2))
		{
			printf("Manager: Dropping message from process %d to process %d\n",	from_process_id, to_process_id);
			deleted_channel = a_channel;
			num_messages_dropped++;
			break;
		}
	}

	temp_channels->erase(deleted_channel);
*/
}

// ----------------------------------------------------------------------------

void SN_Manager::Send_Messages()
{
	DEBUG_2("Manager: Delivering messages for round %d...\n", round);

	map<Link,SN_Message>::iterator a_channel;

	for(a_channel = channels.begin(); a_channel != channels.end(); a_channel++)
	{
		Link from_to = a_channel->first;

		int from_process_id = from_to.first;
		int to_process_id = from_to.second;
		SN_Message outgoing_message = a_channel->second;

		SN_Packet outgoing_packet;
		outgoing_packet.type = SN_Packet::PROCESS_MSG;
		outgoing_packet.from = from_process_id;
		outgoing_packet.to = to_process_id;
		outgoing_packet.message = outgoing_message;

		DEBUG_3("Manager: Sending message from process %d to process %d\n",
			from_process_id, to_process_id);

		m->send(to_process_id, &outgoing_packet, sizeof(SN_Packet));
		num_messages_received++;
	}

	channels.clear();
}

// ----------------------------------------------------------------------------

void SN_Manager::End_Round()
{
	DEBUG_2("Manager: signalling end of message to send in round %d...\n", round);

	set<SN_Process *>::iterator a_process;

	for(a_process = processes.begin(); a_process != processes.end(); a_process++)
	{
		SN_Packet end_msg;
		end_msg.type = SN_Packet::END_ROUND_MSG;
		end_msg.from = SN_Manager::MANAGER_PROCESS_ID;
		end_msg.to = (*a_process)->process_id;

		m->send((*a_process)->process_id, &end_msg, sizeof(SN_Packet));
	}
}

// ----------------------------------------------------------------------------

void SN_Manager::End_Simulation()
{
	printf("\nManager: received signal to end simulation...\n", round);

	printf("\tRounds = %d\n", round);
	printf("\tMessages sent = %d\n", num_messages_sent);
	printf("\tMessages received = %d\n", num_messages_received);
	printf("\tMessages dropped = %d\n", num_messages_dropped);

	exit(0);
}

// ----------------------------------------------------------------------------

void SN_Manager::Insert_Process(SN_Process *new_process)
{
	DEBUG_2("Adding process %d to the manager...\n", new_process->process_id);

	processes.insert(new_process);
	
	// Make the infrastructure call us, so we can then call the process
	MS_Regnode(new_process->process_id,SN_Manager::Start_Sim_Process, NULL);
}
