#include "sn_debug.h"
#include "sn_process.h"
#include <stdio.h>
#include "sn_packet.h"
#include "sn_manager.h"

SN_Process::SN_Process()
{
	// Each time a new process is created, it gets the next process id for the
	// infrastructure
	static next_process_id = 1;

	process_id = next_process_id;
	next_process_id++;
}

// ----------------------------------------------------------------------------

void SN_Process::Run(MS_Msgbox *in_m)
{
	// Storing pointer is better than storing Messages because the set<> template
	// requires a < operator. If we store Message *'s then the user doesn't have
	// to write a < operator for the Message class. This means we have to create
	// and destroy message objects though. :(
	map<SN_Message *,int> incoming_messages;

	DEBUG_1("Process %d: Started.\n", process_id);

	m = in_m;

	while(1)
	{
		SN_Packet incoming_packet;

		// Messages will always come from the manager. The client has to look inside
		// the SN_Packet::from to see the sender's process ID.
		int manager_proc_id;

		m->recv_any(&incoming_packet, sizeof(SN_Packet),&manager_proc_id);

		if (incoming_packet.type == SN_Packet::START_ROUND_MSG)
		{
			DEBUG_2("Process %d: Received start round message\n", process_id);

			Generate_Messages();
		}
		else if (incoming_packet.type == SN_Packet::PROCESS_MSG)
		{
			DEBUG_2("Process %d: Received process message\n", process_id);

			// Create a new message object and store a pointer to it in
			// incoming_messages
			SN_Message* inserted_message = new SN_Message;
			*inserted_message = incoming_packet.message;

			incoming_messages.insert(make_pair(inserted_message,incoming_packet.from));
		}
		else if (incoming_packet.type == SN_Packet::END_ROUND_MSG)
		{
			DEBUG_2("Process %d: Received end round message\n",process_id);
			Transition(incoming_messages);

			// Delete each message object in incoming_messages
			map<SN_Message *,int>::iterator a_message;

			for(a_message = incoming_messages.begin();
				a_message != incoming_messages.end(); a_message++)
				delete a_message->first;

			// clear the pointers
			incoming_messages.clear();
		}
		else
		{
			DEBUG("Unknown message type!\n");
			exit (1);
		}
	}
}

// ----------------------------------------------------------------------------

void SN_Process::_Send_Message(SN_Message *outbound_msg, int to_process_id)
{
	SN_Packet outgoing_packet;

	outgoing_packet.from = process_id;
	outgoing_packet.to = to_process_id;
	outgoing_packet.message = *outbound_msg;
	outgoing_packet.type = SN_Packet::PROCESS_MSG;

	m->send(SN_Manager::MANAGER_PROCESS_ID, &outgoing_packet, sizeof(SN_Packet));
}

// ----------------------------------------------------------------------------

void SN_Process::Send(SN_Message *outbound_msg,int to_process_id)
{
	DEBUG_2("Process %d: Sending message to process %d\n", process_id, to_process_id);

	_Send_Message(outbound_msg,to_process_id);

	Send_Finished();
}

// ----------------------------------------------------------------------------

void SN_Process::Send(map<SN_Message *,int> *outbound_msgs)
{
	DEBUG_2("Process %d: Sending a set of %d messages\n", process_id, outbound_msgs->size());

	map<SN_Message *,int>::iterator a_message_pair;

	for(a_message_pair = outbound_msgs->begin(); a_message_pair != outbound_msgs->end();
		a_message_pair++)
	{
		DEBUG_3("Process %d: Sending message to process %d\n", process_id, a_message_pair->second);

		_Send_Message(a_message_pair->first,a_message_pair->second);
	}

	Send_Finished();
}

// ----------------------------------------------------------------------------

void SN_Process::Send_Finished()
{
	SN_Packet outgoing_packet;

	outgoing_packet.from = process_id;
	outgoing_packet.to = SN_Manager::MANAGER_PROCESS_ID;
	outgoing_packet.type = SN_Packet::DONE_SENDING_MSG;

	DEBUG_3("Process %d: Sending \"finished sending\" message to manager\n", process_id);

	m->send(SN_Manager::MANAGER_PROCESS_ID, &outgoing_packet, sizeof(SN_Packet));
}
