<?php
/********
Plugin Name: Category Order
Plugin URI: http://www.coppit.org/code/
Description: Sorts the sidebar category list according to a user-defined 'Category Order' option in the 'Manage' menu 
Version: 2.0.1
Author: David Coppit
Author URI: http://www.coppit.org/
*********/

//==========================================================================

/*
Notes on options, for developers:

- Category_Order_Put_Post_Count_Inside_Link: A Boolean that indicates whether
  the number of posts "(8)" should be inside the link for the category
- Category_Order: A serialized data structure that represents the order,
  nesting, and spacing of categories.

This plugin automatically creates defaults for these options. For the first,
the lack of an option is equivalent to "false". For the second, the plugin
will query the database of categories, adding them in alphabetical order to
the data structure at the top level.

The plugin always checks the database, removing or adding categories as
necessary.
*/

/*
Notes on the category order data structure, for developers:

If this is the category structure that the user wants:

  1
  - 1.1
  2
  - 2.1
  - 2.2
  3
  [A space]
  4
  - 4.1

The associated data structure is identical to the one produced by the
following code:

  $category_order = array();
  $category_order[0] = array('1',0);
  $category_order[1] = array('1.1',1);
  $category_order[2] = array('2',0);
  $category_order[3] = array('2.1',1);
  $category_order[4] = array('2.2',1);
  $category_order[5] = array('3',0);
  $category_order[6] = array('',0);
  $category_order[7] = array('4',0);
  $category_order[8] = array('4.1',1);

The category_order hash maps parents to their ordered children. '1', '2', etc.
are the category IDs. 
*/

//==========================================================================

include_once('category-order-management.php');
include_once('category-order-shared.php');

$debug_category_order = false;

class category_order {

// We assume that $original_html is valid HTML, with <li> or <br /> for each
// list item. We ignore any sublists, overriding them with ours.
function sort_categories($original_html) {

//$original_html = '<li>No categories</li>';

	// Skip unless it looks like we were called with the entire list, or if the
	// list is empty.
	if (strpos($original_html, '<') === FALSE ||
			strstr($original_html, __("No categories")) !== FALSE)
		return $original_html;

//print "vvvvvvvvvv\n"; category_order::print_html_data($original_html);print "^^^^^^^^^^\n";

	if (get_bloginfo('version') >= 2.1 &&
			preg_match('/(<li class="categories">.*?<ul>\s*)(.*)(<\/ul.*)/si',$original_html,$matches))
	{
		$prefix = $matches[1];
		$processed_html = $matches[2];
		$suffix = $matches[3];
	} else {
		$prefix = '';
		$processed_html = $original_html;
		$suffix = '';
	}

	$id_to_item = category_order::extract_items($processed_html);
//category_order::print_html_data($id_to_item);

	if (is_null($id_to_item))
		return $original_html;

	$new_html = category_order::build_html($id_to_item, $prefix, $suffix);

	return $new_html;
}

// --------------------------------------------------------------------

// Parses the HTML given to us by Wordpress, extracting the
// link/post-count/etc from it. Returns a hash mapping the category id to
// this information.
function extract_items($original_html) {
	// Remove all <h2>, <ul>, </ul>, and </li>
	$processed_html = preg_replace('/(<\s*(ul|h\d)[^>]*>|<\/\s*ul\s*>)/i',
		'',$original_html);

	// Remove double <li> <li> resulting from sub-lists
	$processed_html = preg_replace('/<\s*li\s*>\s*(<\s*li\s*>\s*)/i',
		'$1',$processed_html);
	$processed_html = preg_replace('/(<\s*\/\s*li\s*>\s*)<\s*\/\s*li\s*>\s*/i',
		'$1',$processed_html);

	$processed_html = preg_replace('/(<\s*\/\s*li\s*>\s*)<\s*\/\s*li\s*>\s*/i',
		'$1',$processed_html);

	// Split up the list, then remove all empty items and clean up the HTML a
	// bit. Empty items can occur if <ul> is removed for a sublist, leaving
	// <li>\s*<li>. It can also occur at the start of the list after the split.
	// Also remove any whitespace before or after.
	if (preg_match('/<\s*li/i',$processed_html)) {
		$items = preg_split('/<\s*li\b/i', $processed_html);

		for ($i=0; $i < count($items); $i++) {
			$items[$i] = '<li' . $items[$i];

			$items[$i] = preg_replace('/\n/',' ',$items[$i]);
			$items[$i] = preg_replace('/^\s*(.*?)\s*$/s','$1',$items[$i]);

			if (preg_match('/^<\s*li$/',$items[$i])) {
				array_splice($items,$i,1);
				$i--;
				continue;
			}

			// Add </li> missing as a result of sub-lists
			if (preg_match('/<\s*\/\s*li/i',$items[$i]) == 0)
				$items[$i] .= '</li>';
		}
	} elseif (preg_match('/<\s*br\s*\/>/i',$processed_html)) {
		$items = preg_split('/<\s*br\s*\/>/i', $processed_html);

		for ($i=0; $i < count($items); $i++) {
			$items[$i] = $items[$i] . '<br/>';

			$items[$i] = preg_replace('/\n/',' ',$items[$i]);
			$items[$i] = preg_replace('/^\s*(.*?)\s*$/s','$1',$items[$i]);

			if (preg_match('/^<\s*br[^>]*>$/',$items[$i])) {
				array_splice($items,$i,1);
				$i--;
			}
		}
	} else {
		return null;
	}

	// Move the count inside the link
	if (get_option('Category_Order_Put_Post_Count_Inside_Link')) {
		for ($i=0; $i < count($items); $i++)
			$items[$i] =
				preg_replace('/\s*<\/a>\s*(\(\d+\))/i',' $1</a>',$items[$i]);
	}

	global $wp_rewrite;
	$category_permastruct = $wp_rewrite->get_category_permastruct();

	if (empty($category_permastruct)) {
		$using_category_id = true;
		$category_pattern =
			'/' . preg_quote(get_settings('home') . '/?cat=', '/') . '(\d+)/';
	} else {
		$using_category_id = false;
		$category_pattern = '/' . preg_quote(get_settings('home'), '/') .
			'.*?' . preg_quote($category_permastruct, '/') . '/';
		$category_pattern =
			preg_replace('/%category%/', '(?:[^\/"]+\/)*([^\/"]+)', $category_pattern);
	}

//$category_pattern = '/http\:\/\/rioapartments123\.com\/blog.*?\/category\/(?:[^\/"]+\/)*([^\/"]+)/';

	// Make a hash for easy lookup
	$id_to_item = array();
	foreach ($items as $item) {
		$matched = (preg_match($category_pattern,$item,$matches) > 0);

		global $debug_category_order;

		if (!$matched || $debug_category_order) {
			print '<div style="position:absolute;top:0;left:0;background-color:yellow;width:body.clientWidth;color:black;">
			';
			if (!$matched)
				print '
<div style="position:absolute;top:0;left:0;background-color:yellow;width:body.clientWidth;color:black;">
<p>Category Order could not understand your category HTML. Please do the
following:</p>
<ul style="color:black;list-style-type:disc;margin-left:1em;">
<li> Disable all plugins except for category order to see if some other plugin
is causing the problem. If the problem goes away, re-enable each plugin until
you find the incompatible one.
<li> Try switching to a different theme, like the WordPress default theme.
</ul>
<p>Once you have tried the above steps, email <a
href="mailto:david@coppit.rg">david@coppit.org</a> with the results of these
debugging steps. Also include the following information:</p>
			';
			print "<p>Original HTML:<br>\n";
			category_order::print_html_data($original_html);
			print "<p>Processed HTML:<br>\n";
			category_order::print_html_data($processed_html);
			print '</p><p>Category pattern:<br>';
			category_order::print_html_data($category_pattern);
			print '</p><p>Items:<br>';
			category_order::print_html_data($items);
			print "</p></div>\n";
			return null;
		}

		if ($using_category_id) {
			$id = $matches[1];
		} else {
			$category = category_order::get_category_by_slug($matches[1]);
			$id = $category->cat_ID;
		}

		$id_to_item[$id] = $item;
	}

	return $id_to_item;
}

// --------------------------------------------------------------------

function get_category_by_slug($slug) {
	$categories = get_categories();

	$slug_category = null;
	$warn = false;

	foreach ($categories as $category)
		if ($category->category_nicename === $slug)
		{
			if (!is_null($slug_category))
				$warn = true;

			$slug_category = $category;
		}

	if ($warn)
	{
		print '<div style="position:absolute;top:0;left:0;background-color:yellow;width:body.clientWidth;color:black;">

<p>You are using a category link style that uses a category slug rather than a
category ID, and you have two categories with the same category slug. Category
Order cannot reorder your categories properly because the links in the
category HTML below are not unique. You need to either use a link type that
has IDs, or make your slugs unique to their categories.</p>
';
		category_order::print_html_data($original_html);
		print "</p></div>\n";
	}

	return $slug_category;
}

// --------------------------------------------------------------------

function build_html($id_to_item, $prefix, $suffix) {
//category_order::print_html_data($category_order);
	unset($last_nonspacer_index);
	$number_of_ending_spacers = 0;

	for ($i=count($category_order)-1; $i >= 0; $i--) {
//print $category_order[$i][0] . "::" . $id_to_item[$category_order[$i][0]] . "<br>\n";
		# In case the category was filtered out in wp_list_cats
		if ($id_to_item[$category_order[$i][0]] == '')
			continue;

		if ($category_order[$i][0] == '')
			$number_of_ending_spacers++;
		else {
			$last_nonspacer_index = $i;
			break;
		}
	}

	$category_order = category_order_shared::get_category_order();

  if (get_option('Category_Order_Enable_Category_Folding'))
		category_order::fold_categories($id_to_item,$category_order);

	if (preg_match('/<\s*li/i',$id_to_item[$category_order[0][0]]))
		$html = category_order::build_html_list($id_to_item,$category_order,
			$last_nonspacer_index,$number_of_ending_spacers);
	else
		$html = category_order::build_html_br($id_to_item,$category_order,
			$last_nonspacer_index,$number_of_ending_spacers);

	return $prefix.$html.$suffix;
}

// --------------------------------------------------------------------

function fold_categories(&$id_to_item,$category_order) {
	global $wp_query;

	$current_cat = null;
	
	if (!is_page())
		$current_cat = $wp_query->get_queried_object_id();

	for ($i=0; $i < count($category_order); $i++) {
		// Always show the top-level categories
		if ($category_order[$i][1] == 0)
			continue;

		// Always show the selected category
		if ($category_order[$i][0] == $current_cat)
			continue;

		// Make immediate subcategories visible, as well as initial subcategories
		// that have no parent
		if (!is_null($current_cat))
		{
			$is_visible_child = true;

			for ($j=$i-1; $j >= 0; $j--)
			{
				if ($category_order[$j][0] == '')
					continue;
					
				if ($category_order[$j][1] < $category_order[$i][1]) 
				{
					if ($category_order[$j][0] != $current_cat)
						$is_visible_child = false;
					break;
				}
			}

			if ($is_visible_child)
				continue;
		}

		// Make ancestors visible
		if (!is_null($current_cat))
		{
			$is_visible_parent = false;

			for ($j=$i+1; $j < count($category_order); $j++)
			{
				if ($category_order[$j][0] == '')
					continue;
						
				if ($category_order[$j][1] <= $category_order[$i][1]) 
					break;
				else
				{
					if ($category_order[$j][0] == $current_cat)
					{
						$is_visible_parent = true;
						break;
					}
				}
			}
		}

		// Make siblings of parents visible
		if (!is_null($current_cat))
		{
			$is_visible_uncle = false;

			for ($j=$i+1; $j < count($category_order); $j++)
			{
				if ($category_order[$j][0] == '')
					continue;
						
				if ($category_order[$j][1] < $category_order[$i][1]) 
					break;
				else
				{
					if ($category_order[$j][0] == $current_cat)
					{
						$is_visible_uncle = true;
						break;
					}
				}
			}

			for ($j=$i-1; $j >= 0; $j--)
			{
				if ($category_order[$j][0] == '')
					continue;
						
				if ($category_order[$j][1] < $category_order[$i][1]) 
					break;
				else
				{
					if ($category_order[$j][0] == $current_cat)
					{
						$is_visible_uncle = true;
						break;
					}
				}
			}

			if ($is_visible_uncle)
				continue;
		}


		// Save the post count for later.
		$post_count = 0;
		
		if (preg_match('/\s*<\/a>\s*\((\d+)\)/i',
				$id_to_item[$category_order[$i][0]],$matches))
			$post_count = $matches[1];


		unset($id_to_item[$category_order[$i][0]]);


		// Update the post count.
		if (get_option('Category_Order_Sum_Post_Counts'))
		{
			for ($j=$i-1; $j >= 0; $j--)
			{
				if ($category_order[$j][0] == '')
					continue;
					
				if ($category_order[$j][1] < $category_order[$i][1]) 
				{
					if (preg_match('/\s*<\/a>\s*\((\d+)\)/i',
							$id_to_item[$category_order[$j][0]],$matches))
					{
						$new_count = $matches[1] + $post_count;
						$id_to_item[$category_order[$j][0]] =
							preg_replace('/(\s*<\/a>\s*\()(\d+)\)/i',"\${1}$new_count)",
							$id_to_item[$category_order[$j][0]]);
					}

					break;
				}
			}
		}


		// Put a "+" to indicate that you can expand the link.
		for ($j=$i-1; $j >= 0; $j--)
		{
			if ($category_order[$j][0] == '' ||
					$id_to_item[$category_order[$j][0]] == '')
				continue;

			if ($category_order[$j][1] < $category_order[$i][1]) 
			{
				if (preg_match('/<\s*li/i',$id_to_item[$category_order[$j][0]]))
					$id_to_item[$category_order[$j][0]] =
						preg_replace("/(<\s*li[^>]*>)(?!\+)/i", '$1+', $id_to_item[$category_order[$j][0]] );
				else
					$id_to_item[$category_order[$j][0]] = 
						"+" . $id_to_item[$category_order[$j][0]];
				break;
			}
		}
	}
}

// --------------------------------------------------------------------

function build_html_list($id_to_item,$category_order,$last_nonspacer_index,
		$number_of_ending_spacers) {
	$html = '';

	$number_of_leading_spacers = 0;

	for ($i=0; $i < count($category_order); $i++) {
		if ($category_order[$i][0] == '') {
			$number_of_leading_spacers++;
			continue;
		}

		if ($id_to_item[$category_order[$i][0]] == '')
			continue;

		if ($number_of_leading_spacers == 0 &&
				$category_order[$i][1] == 0 &&
				(!isset($last_nonspacer_index) ||
				 $i != $last_nonspacer_index ||
				 $number_of_ending_spacers == 0)) {
			$html .= $id_to_item[$category_order[$i][0]] . "\n";
			continue;
		}

		$id_to_item[$category_order[$i][0]] =
			preg_replace("/<\s*li\s*/i", "", $id_to_item[$category_order[$i][0]] );

		$existing_style = '';

		if (preg_match('/[^>]*(style\s*=\s*[\'"](.*?)[\'"])/i',
				$id_to_item[$category_order[$i][0]], $matches)) {
			$existing_style = $matches[2];

			$id_to_item[$category_order[$i][0]] =
				preg_replace('/' . $matches[1] . '/', "",
				$id_to_item[$category_order[$i][0]] );
		}

		$html .= "<li style=\"$existing_style";

		if ($number_of_leading_spacers != 0)
			$html .= "margin-top:${number_of_leading_spacers}em;";

		if ($category_order[$i][1] != 0)
			$html .= 'margin-left:' . $category_order[$i][1] . 'em;';

		if (isset($last_nonspacer_index) && $i == $last_nonspacer_index &&
				$number_of_ending_spacers != 0)
			$html .= "margin-bottom:${number_of_ending_spacers}em;";

		$html .= '"' . $id_to_item[$category_order[$i][0]] . "\n";

		$number_of_leading_spacers = 0;
	}

	return $html;
}

// --------------------------------------------------------------------

function build_html_br($id_to_item,$category_order,$last_nonspacer_index,
		$number_of_ending_spacers) {
	$html = "<div style=\"margin-top:0em;margin-bottom:0em;\">\n";

	$number_of_leading_spacers = 0;

	for ($i=0; $i < count($category_order); $i++) {
		if ($id_to_item[$category_order[$i][0]] == '')
			continue;

		if ($category_order[$i][0] == '') {
			$number_of_leading_spacers++;
			continue;
		}

		if ($number_of_leading_spacers == 0 &&
				$category_order[$i][1] == 0 &&
				(!isset($last_nonspacer_index) ||
				 $i != $last_nonspacer_index ||
				 $number_of_ending_spacers == 0)) {
			$html .= "<div> " . $id_to_item[$category_order[$i][0]] . " </div>\n";
			continue;
		}

		$id_to_item[$category_order[$i][0]] =
			preg_replace("/<\s*br[^>]*>\s*/i", "", $id_to_item[$category_order[$i][0]] );


		$html .= '<div style="';

		if ($number_of_leading_spacers != 0)
			$html .= "margin-top:${number_of_leading_spacers}em;";

		if ($category_order[$i][1] != 0)
			$html .= 'margin-left:' . $category_order[$i][1] . 'em;';

		if (isset($last_nonspacer_index) && $i == $last_nonspacer_index &&
				$number_of_ending_spacers != 0)
			$html .= "margin-bottom:${number_of_ending_spacers}em;";

		$html .= '"> ' . $id_to_item[$category_order[$i][0]] . " </div>\n";

		$number_of_leading_spacers = 0;
	}

	$html .= "</div>\n";

	return $html;
}

// --------------------------------------------------------------------

// Used during development to dump a data structure to html
function print_html_data($data) {
	$string = htmlspecialchars(print_r($data,1));
	$string = preg_replace("/\n/", "<br>\n", $string);
	$string = preg_replace("/ /", "&nbsp;", $string);

	print $string;
}

// --------------------------------------------------------------------

function backtrace()
{
   $output = "<div style='text-align: left; font-family: monospace;'>\n";
   $output .= "<b>";
   $output .= _e('Backtrace:', 'category-access');
   $output .= "</b><br />\n";
   $backtrace = debug_backtrace();

   foreach ($backtrace as $bt) {
       $args = '';
       foreach ($bt['args'] as $a) {
           if (!empty($args)) {
               $args .= ', ';
           }
           switch (gettype($a)) {
           case 'integer':
           case 'double':
               $args .= $a;
               break;
           case 'string':
               $a = htmlspecialchars(substr($a, 0, 64)).((strlen($a) > 64) ? '...' : '');
               $args .= "\"$a\"";
               break;
           case 'array':
               $args .= 'Array('.count($a).')';
               break;
           case 'object':
               $args .= 'Object('.get_class($a).')';
               break;
           case 'resource':
               $args .= 'Resource('.strstr($a, '#').')';
               break;
           case 'boolean':
               $args .= $a ? 'True' : 'False';
               break;
           case 'NULL':
               $args .= 'Null';
               break;
           default:
               $args .= 'Unknown';
           }
       }
       $output .= "<br />\n";
       $output .= "<b>file:</b> {$bt['line']} - {$bt['file']}<br />\n";
       $output .= "<b>call:</b> {$bt['class']}{$bt['type']}{$bt['function']}($args)<br />\n";
   }
   $output .= "</div>\n";
   return $output;
}
}

// Make sure this is higher than Category Access' 10000, so that it will
// process the categories after they've been filtered out.

add_action('wp_list_categories', array('category_order','sort_categories'), 11000);

// For versions prior to 2.1
add_action('list_cats', array('category_order','sort_categories'), 11000);

?>
